// Copyright (C) 2025, Edensoft Apps
// Full notice in NOTICE.txt file you should have received along with this file

#ifndef ED_ARRAY_HPP
#define ED_ARRAY_HPP

#include "IM_ContainerSolidFinite.hpp"
#include "EachFromArgs.hpp"
#include "TypeVariationsConcepts.hpp"
#include "edeniumConfig_debugModeSettings.hpp"
#include "Memory.hpp"

#include <cstdint>
#include <concepts>
#include <memory>

namespace ed
{
   template <typename StoredType, uintmax_t Size>
   class Array :public IM_ContainerSolidFinite<StoredType>
   {
      static_assert(Size != 0, "`Size` template argument must not be 0.");

      public:

         /// Constructors

            Array()                                              requires(std::constructible_from<StoredType>);
            Array(const Array<StoredType, Size>&  arg)           requires(std::constructible_from<StoredType, const StoredType&>);
            Array(      Array<StoredType, Size>&& arg) noexcept;

            template <typename FirstArgType, typename... RestArgTypes>
               explicit Array(ed::EachFromArgs tag, FirstArgType firstArgForEachElem, RestArgTypes... restArgsForEachElem)
                  requires(std::constructible_from<StoredType, FirstArgType, RestArgTypes...>);

         /// Destructor

            virtual ~Array();

         /// Methods

            void      repoint(      StoredType*& pointer, const uintmax_t& currentIndex, const uintmax_t& targetIndex)                override;
            void      repoint(const StoredType*& pointer, const uintmax_t& currentIndex, const uintmax_t& targetIndex) const          override;
            uintmax_t size   ()                                                                                        const noexcept override;

            template <POINTER_TO_CONST_AGNOSTIC<StoredType> AdequatePtrType> void repointForward   (AdequatePtrType& pointer, const uintmax_t& offset) const noexcept;
            template <POINTER_TO_CONST_AGNOSTIC<StoredType> AdequatePtrType> void repointBackward  (AdequatePtrType& pointer, const uintmax_t& offset) const noexcept;
            template <POINTER_TO_CONST_AGNOSTIC<StoredType> AdequatePtrType> void repointToNext    (AdequatePtrType& pointer)                          const noexcept;
            template <POINTER_TO_CONST_AGNOSTIC<StoredType> AdequatePtrType> void repointToPrevious(AdequatePtrType& pointer)                          const noexcept;

         /// Operator overloads

            const StoredType& operator() (const uintmax_t& index) const override;
                  StoredType& operator() (const uintmax_t& index)       override;

            Array<StoredType, Size>& operator= (      Array<StoredType, Size>&& arg) noexcept;
            Array<StoredType, Size>& operator= (const Array<StoredType, Size>&  arg)           requires(std::copyable<StoredType>);

      private:

         /// Methods

            template <POINTER_TO_CONST_AGNOSTIC<StoredType> AdequatePtrType>
               void m_repointGeneric(AdequatePtrType& pointer, const uintmax_t& currentIndex, const uintmax_t& targetIndex) const noexcept(!config::debuggingEnabledFor_Array_repoint);

         /// Variables

            std::unique_ptr<StoredType[], decltype(&dealloc)> m_dataPtr;
   };
}

#include "Array_xtors.tpp"
#include "Array_methods.tpp"
#include "Array_op_overloads.tpp"

#endif // ED_ARRAY_HPP
